/*
  Web Serial untuk ESP32/Arduino, disesuaikan dengan log mulaibaru.ino
  - Koneksi serial di browser (Chrome/Edge)
  - Parsing baris: JSON, KV, serta pola log khusus (WiFi, mode, relay, alarm)
*/

  const ui = {
  baudRate: document.getElementById('baudRate'),
  parserMode: document.getElementById('parserMode'),
  connectBtn: document.getElementById('connectBtn'),
  disconnectBtn: document.getElementById('disconnectBtn'),
  sendInput: document.getElementById('sendInput'),
  sendBtn: document.getElementById('sendBtn'),
  defrostBtn: document.getElementById('defrostBtn'),
  status: document.getElementById('status'),
  parsed: document.getElementById('parsed'),
  log: document.getElementById('log'),
  downloadBtn: document.getElementById('downloadBtn'),
  // Telemetry card
  t1Val: document.getElementById('t1Val'),
  t2Val: document.getElementById('t2Val'),
  t3Val: document.getElementById('t3Val'),
  modeVal: document.getElementById('modeVal'),
  ipVal: document.getElementById('ipVal'),
  rl1Val: document.getElementById('rl1Val'),
  rl2Val: document.getElementById('rl2Val'),
  rl3Val: document.getElementById('rl3Val'),
  lastUpdate: document.getElementById('lastUpdate'),
  espStatus: document.getElementById('espStatus'),
  modeDuration: document.getElementById('modeDuration'),
  // Pengaturan batas
  limitR: document.getElementById('limitR'),
  limitE: document.getElementById('limitE'),
    limitO: document.getElementById('limitO'),
    applyLimitsBtn: document.getElementById('applyLimitsBtn'),
    applyConfirmModal: document.getElementById('applyConfirmModal'),
    applyConfirmR: document.getElementById('applyConfirmR'),
    applyConfirmE: document.getElementById('applyConfirmE'),
    applyConfirmO: document.getElementById('applyConfirmO'),
    applyConfirmYes: document.getElementById('applyConfirmYes'),
    applyConfirmNo: document.getElementById('applyConfirmNo'),
    applyConfirmProgress: document.getElementById('applyConfirmProgress'),
    // Modal DEFROS
    defrostConfirmModal: document.getElementById('defrostConfirmModal'),
    defrostConfirmYes: document.getElementById('defrostConfirmYes'),
    defrostConfirmNo: document.getElementById('defrostConfirmNo'),
    defrostConfirmProgress: document.getElementById('defrostConfirmProgress'),
  // XMPP
  xmppJid: document.getElementById('xmppJid'),
  xmppPassword: document.getElementById('xmppPassword'),
  xmppWsUrl: document.getElementById('xmppWsUrl'),
  xmppRecipient: document.getElementById('xmppRecipient'),
  xmppConnectBtn: document.getElementById('xmppConnectBtn'),
  xmppDisconnectBtn: document.getElementById('xmppDisconnectBtn'),
  xmppSendTelemetryBtn: document.getElementById('xmppSendTelemetryBtn'),
  xmppStatus: document.getElementById('xmppStatus'),
  // Histori
  historyTbody: document.getElementById('historyTbody'),
  historyRefreshBtn: document.getElementById('historyRefreshBtn'),
};

// Konfigurasi base URL backend (harus sama dengan yang dipakai ESP32 di SERVER_URL)
// Ubah jika diperlukan. Bisa di-override lewat window.BACKEND_BASE_URL dari index.html.
const BACKEND_BASE_URL = window.BACKEND_BASE_URL || 'mulaibaru/backend/';

let port = null;
let reader = null;
let readBuffer = '';
let keepReading = false;
let writer = null;
let logs = [];
let xmppClient = null;


const state = {
  t1: null,
  t2: null,
  t3: null,
  mode: null, // RUN/DEF
  ip: null,
  rl1: null,
  rl2: null,
  rl3: null,
  alarm: false,
  limits: { R: null, E: null, O: null },
  lastAt: null,
  modeChangedAt: null,
};

// Timestamp terakhir update limits dari backend untuk sinkronisasi multi-klien
let limitsLastUpdated = null;

function setStatus(text) { if (ui.status) ui.status.textContent = `Status: ${text}`; }
function getParserMode() { return ui.parserMode?.value || 'auto'; }
function setXMPPStatus(text) { if (ui.xmppStatus) ui.xmppStatus.textContent = `XMPP: ${text}`; }

function updateMode(newMode) {
  if (!newMode) return;
  const nm = String(newMode).toUpperCase() === 'DEF' ? 'DEF' : 'RUN';
  if (state.mode !== nm) {
    state.mode = nm;
    state.modeChangedAt = Date.now();
    saveModeToStorage();
  }
}

function formatDuration(ms) {
  if (!Number.isFinite(ms) || ms < 0) return '-';
  const totalSec = Math.floor(ms / 1000);
  const h = Math.floor(totalSec / 3600);
  const m = Math.floor((totalSec % 3600) / 60);
  const s = totalSec % 60;
  const pad = (n) => String(n).padStart(2, '0');
  return h > 0 ? `${pad(h)}:${pad(m)}:${pad(s)}` : `${pad(m)}:${pad(s)}`;
}

function updateModeDurationUI() {
  if (!ui.modeDuration) return;
  if (!state.mode || !state.modeChangedAt) { ui.modeDuration.textContent = '-'; return; }
  const diff = Date.now() - state.modeChangedAt;
  const cls = state.mode === 'DEF' ? 'mode-def' : 'mode-run';
  ui.modeDuration.innerHTML = `<span class="${cls}">${state.mode}</span> ${formatDuration(diff)}`;
}

// Simpan/pulihkan mode ke/dari localStorage agar durasi bertahan saat refresh
function saveModeToStorage() {
  try {
    localStorage.setItem('mode', state.mode || '');
    localStorage.setItem('modeChangedAt', state.modeChangedAt != null ? String(state.modeChangedAt) : '');
  } catch (_) {}
}

function loadModeFromStorage() {
  try {
    const m = localStorage.getItem('mode');
    const atStr = localStorage.getItem('modeChangedAt');
    const atNum = atStr ? Number(atStr) : null;
    if (m && (m === 'RUN' || m === 'DEF') && Number.isFinite(atNum)) {
      state.mode = m;
      state.modeChangedAt = atNum;
    }
  } catch (_) {}
}

// Normalisasi angka dari JSON/KV: dukung string dan koma desimal
function normalizeNumber(val) {
  if (val == null) return null;
  if (typeof val === 'number' && Number.isFinite(val)) return val;
  if (typeof val === 'string') {
    const s = val.replace(',', '.').trim();
    const n = Number(s);
    return Number.isFinite(n) ? n : null;
  }
  return null;
}

// Stepper handler: sesuaikan nilai input number via tombol +/−
function adjustInput(targetId, dir) {
  const input = document.getElementById(targetId);
  if (!input) return;
  const stepRaw = input.step || '1';
  const step = Number(String(stepRaw).replace(',', '.'));
  const delta = Number.isFinite(step) ? step : 1;
  const cur = normalizeNumber(input.value);
  let next = (cur != null ? cur : 0) + dir * delta;
  const min = input.min !== '' ? Number(input.min) : null;
  const max = input.max !== '' ? Number(input.max) : null;
  if (min != null && Number.isFinite(min) && next < min) next = min;
  if (max != null && Number.isFinite(max) && next > max) next = max;
  input.value = Number(next.toFixed(1)).toFixed(1);
}

function appendLog(line) {
  if (!line) return;
  logs.push(line);
  if (logs.length > 5000) logs.shift();
  ui.log.textContent += line + '\n';
  ui.log.scrollTop = ui.log.scrollHeight;
  ui.downloadBtn.disabled = logs.length === 0;
}

function showParsed(obj) {
  try { ui.parsed.textContent = JSON.stringify(obj, null, 2); }
  catch (_) { ui.parsed.textContent = String(obj); }
}

function renderTelemetry() {
  ui.t1Val.textContent = state.t1 == null ? '-' : `${state.t1}`;
  ui.t2Val.textContent = state.t2 == null ? '-' : `${state.t2}`;
  ui.t3Val.textContent = state.t3 == null ? '-' : `${state.t3}`;
  ui.modeVal.textContent = state.mode || '-';
  if (ui.modeVal) {
    ui.modeVal.classList.remove('mode-def', 'mode-run');
    if (state.mode === 'DEF') ui.modeVal.classList.add('mode-def');
    else if (state.mode === 'RUN') ui.modeVal.classList.add('mode-run');
  }
  if (ui.ipVal) ui.ipVal.textContent = state.ip || '-';
  ui.rl1Val.textContent = state.rl1 == null ? '-' : String(state.rl1);
  ui.rl2Val.textContent = state.rl2 == null ? '-' : String(state.rl2);
  ui.rl3Val.textContent = state.rl3 == null ? '-' : String(state.rl3);
  // RL2 infer: heater defrost ON saat DEFROST
  if (state.mode === 'DEF') ui.rl2Val.textContent = '1';
  if (state.mode === 'RUN') ui.rl2Val.textContent = '0';

  // Pewarnaan berdasarkan batas
  [ui.t1Val, ui.t2Val, ui.t3Val].forEach(el => { el.classList.remove('ok', 'bad'); });
  if (state.limits.R != null && state.t1 != null) ui.t1Val.classList.add(state.t1 >= state.limits.R ? 'bad' : 'ok');
  if (state.limits.E != null && state.t2 != null) ui.t2Val.classList.add(state.t2 >= state.limits.E ? 'bad' : 'ok');
  if (state.limits.O != null && state.t3 != null) ui.t3Val.classList.add(state.t3 >= state.limits.O ? 'bad' : 'ok');

  if (ui.lastUpdate) ui.lastUpdate.textContent = new Date().toLocaleTimeString();

  // Indikator ESP32 aktif/tidak berdasarkan waktu update terakhir
  if (ui.espStatus) {
    const now = Date.now();
    const age = state.lastAt ? (now - state.lastAt) : Infinity;
    const isActive = age < 20000; // aktif jika data <= 20 detik yang lalu
    ui.espStatus.textContent = isActive ? 'aktif' : 'tidak aktif';
    ui.espStatus.classList.remove('indicator-ok', 'indicator-bad');
    ui.espStatus.classList.add(isActive ? 'indicator-ok' : 'indicator-bad');
  }

  updateModeDurationUI();

  // Update tombol DEFROST berdasarkan mode
  if (ui.defrostBtn) {
    if (state.mode === 'DEF') {
      ui.defrostBtn.disabled = true;
      ui.defrostBtn.textContent = 'Sedang DEFROST';
      ui.defrostBtn.classList.add('active');
    } else {
      ui.defrostBtn.textContent = 'Manual DEFROS';
      ui.defrostBtn.classList.remove('active');
      ui.defrostBtn.disabled = false; // selalu bisa manual, via Serial atau HTTP
    }
  }
}

// Muat histori temperatur (10 menit, 3 hari terakhir) dari backend
async function loadHistory() {
  let data = null;
  try {
    const resAbs = await fetch(BACKEND_BASE_URL + 'history.php', { cache: 'no-store' });
    if (resAbs.ok) data = await resAbs.json();
  } catch (errAbs) {
    // fallback di bawah
  }
  if (!Array.isArray(data)) {
    try {
      const resLoc = await fetch('mulaibaru/backend/history.php', { cache: 'no-store' });
      if (resLoc.ok) data = await resLoc.json();
    } catch (errLoc) {
      console.warn('Gagal muat histori dari backend:', errLoc);
      return;
    }
  }
  if (!Array.isArray(data)) return;

  // Urutkan terbaru di atas
  try {
    data.sort((a, b) => {
      const ta = new Date(a.at).getTime();
      const tb = new Date(b.at).getTime();
      return tb - ta;
    });
  } catch (_) {}

  const tbody = ui.historyTbody;
  if (!tbody) return;
  tbody.innerHTML = '';

  const fmt = (x) => {
    if (x == null || Number.isNaN(Number(x))) return '-';
    const n = Number(x);
    return Number.isFinite(n) ? n.toFixed(2) : '-';
  };

  let prevHourKey = null;
  for (const item of data) {
    const tr = document.createElement('tr');
    let d = null;
    let timeStr = '-';
    try {
      d = new Date(item.at);
      timeStr = d.toLocaleString();
    } catch (_) {}

    // Tandai awal jam baru untuk memberi garis pembatas antar jam
    let hourKey = null;
    if (d && !isNaN(d.getTime())) {
      const y = d.getFullYear();
      const m = String(d.getMonth() + 1).padStart(2, '0');
      const day = String(d.getDate()).padStart(2, '0');
      const h = String(d.getHours()).padStart(2, '0');
      hourKey = `${y}-${m}-${day} ${h}`;
    }
    if (hourKey && hourKey !== prevHourKey) {
      tr.classList.add('hour-start');
    }

    tr.innerHTML = `
      <td>${timeStr}</td>
      <td>${item.mode ? String(item.mode).toUpperCase() : '-'}</td>
      <td>${fmt(item.t1)}</td>
      <td>${fmt(item.t2)}</td>
      <td>${fmt(item.t3)}</td>
    `;
    tbody.appendChild(tr);
    prevHourKey = hourKey;
  }
}

// Simpan batas telemetri ke backend (POST), dengan fallback absolut->lokal
async function saveLimitsToBackend(r, e, o) {
  const body = new URLSearchParams();
  if (r !== null && !Number.isNaN(r)) body.append('limitR', String(r)); else body.append('limitR', '');
  if (e !== null && !Number.isNaN(e)) body.append('limitE', String(e)); else body.append('limitE', '');
  if (o !== null && !Number.isNaN(o)) body.append('limitO', String(o)); else body.append('limitO', '');
  try {
    const resAbs = await fetch(BACKEND_BASE_URL + 'state.php', {
      method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body, cache: 'no-store'
    });
    if (resAbs.ok) return await resAbs.json();
    throw new Error(`HTTP ${resAbs.status}`);
  } catch (errAbs) {
    try {
      const resLoc = await fetch('mulaibaru/backend/state.php', {
        method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body, cache: 'no-store'
      });
      if (resLoc.ok) return await resLoc.json();
      throw new Error(`HTTP ${resLoc.status}`);
    } catch (errLoc) {
      console.warn('Gagal simpan limit ke backend:', errAbs, errLoc);
      throw errLoc;
    }
  }
}

// Muat batas telemetri dari backend (GET), set ke UI dan state
async function loadLimitsFromBackend() {
  const applyFromObj = (j) => {
    if (!j || !j.limits) return false;
    const R = j.limits.R ?? null;
    const E = j.limits.E ?? null;
    const O = j.limits.O ?? null;
    if (ui.limitR) ui.limitR.value = R != null ? Number(R).toFixed(1) : '';
    if (ui.limitE) ui.limitE.value = E != null ? Number(E).toFixed(1) : '';
    if (ui.limitO) ui.limitO.value = O != null ? Number(O).toFixed(1) : '';
    state.limits = { R, E, O };
    if (j.updated != null) limitsLastUpdated = j.updated;
    renderTelemetry();
    return true;
  };
  try {
    const resAbs = await fetch(BACKEND_BASE_URL + 'state.php', { cache: 'no-store' });
    if (resAbs.ok) {
      const j = await resAbs.json();
      if (applyFromObj(j)) return;
    }
  } catch (errAbs) { console.warn('Gagal muat limits dari backend absolut:', errAbs); }
  try {
    const resLoc = await fetch('mulaibaru/backend/state.php', { cache: 'no-store' });
    if (!resLoc.ok) return;
    const j = await resLoc.json();
    applyFromObj(j);
  } catch (errLoc) { /* abaikan */ }
}

// Terapkan limits jika berbeda atau jika timestamp updated berubah
function applyLimitsIfChanged(j) {
  if (!j || !j.limits) return;
  const updated = j.updated ?? null;
  const R = j.limits.R ?? null;
  const E = j.limits.E ?? null;
  const O = j.limits.O ?? null;
  const changed =
    (updated != null && updated !== limitsLastUpdated) ||
    (state.limits.R !== R || state.limits.E !== E || state.limits.O !== O);
  if (!changed) return;
  limitsLastUpdated = updated ?? limitsLastUpdated;
  state.limits = { R, E, O };
  if (ui.limitR) ui.limitR.value = R != null ? Number(R).toFixed(1) : '';
  if (ui.limitE) ui.limitE.value = E != null ? Number(E).toFixed(1) : '';
  if (ui.limitO) ui.limitO.value = O != null ? Number(O).toFixed(1) : '';
  renderTelemetry();
}

// Polling limits dari backend untuk memastikan sinkronisasi antar klien
async function pollLimitsFromBackend() {
  try {
    const resAbs = await fetch(BACKEND_BASE_URL + 'state.php', { cache: 'no-store' });
    if (resAbs.ok) {
      const j = await resAbs.json();
      applyLimitsIfChanged(j);
      return;
    }
  } catch (errAbs) {
    // abaikan, lanjut ke lokal
  }
  try {
    const resLoc = await fetch('mulaibaru/backend/state.php', { cache: 'no-store' });
    if (resLoc.ok) {
      const j = await resLoc.json();
      applyLimitsIfChanged(j);
    }
  } catch (errLoc) {
    // abaikan kesalahan jaringan lokal
  }
}

function parseKVNumbers(line) {
  // Parse key=value atau key:value untuk angka (t1,t2,t3 dll)
  const result = {};
  let found = false;
  // contoh cocok: t1:25.8 t2:-3.1 t3=28.0
  const regex = /([A-Za-z][\w]*)\s*[:=]\s*(-?\d+(?:\.\d+)?)/g;
  let m;
  while ((m = regex.exec(line)) !== null) {
    const key = m[1];
    const num = Number(m[2]);
    result[key] = num;
    found = true;
  }
  return found ? result : null;
}

function parseSpecialLogs(line) {
  // WiFi IP
  let m = line.match(/WiFi terhubung, IP:\s*(\S+)/);
  if (m) { state.ip = m[1]; }

  // Mode
  if (/Masuk mode DEFROST/i.test(line) || /DEFROST manual dari (WEB|tombol fisik)/i.test(line)) {
    updateMode('DEF');
  }
  if (/DEFROST selesai .* kembali ke RUNNING/i.test(line)) {
    updateMode('RUN');
  }

  // RL ON/OFF: "RL1 -> ON" atau "RL3 -> OFF"
  m = line.match(/RL(\d)\s*->\s*(ON|OFF)/);
  if (m) {
    const idx = Number(m[1]);
    const val = m[2] === 'ON' ? 1 : 0;
    if (idx === 1) state.rl1 = val;
    else if (idx === 2) state.rl2 = val; // jarang dicetak, tapi kalau ada
    else if (idx === 3) state.rl3 = val;
  }

  // Alarm
  if (/ALARM AKTIF/i.test(line)) state.alarm = true;
  if (/Suhu normal, buzzer OFF/i.test(line) || /DEFROST MODE: Alarm dinonaktifkan/i.test(line)) state.alarm = false;
}

function handleLine(line) {
  const mode = getParserMode();
  const trimmed = line.trim();
  if (!trimmed) return;

  appendLog(trimmed);

  // 1) Coba JSON
  if (mode === 'json' || mode === 'auto') {
    try {
      const obj = JSON.parse(trimmed);
      showParsed(obj);
      // Telemetry dari JSON
      const n1 = normalizeNumber(obj.t1);
      const n2 = normalizeNumber(obj.t2);
      const n3 = normalizeNumber(obj.t3);
      if (n1 != null) state.t1 = n1;
      if (n2 != null) state.t2 = n2;
      if (n3 != null) state.t3 = n3;
      if (obj.mode) updateMode(obj.mode); // RUN/DEF
      // Jika durasi tersedia dari ESP32, gunakan untuk menyetel waktu mulai mode
      if (obj.dur != null && Number.isFinite(Number(obj.dur))) {
        const ds = Number(obj.dur);
        if (ds >= 0) state.modeChangedAt = Date.now() - ds * 1000;
        saveModeToStorage();
      }
      // dukung kedua format: rl1..rl4 dan r1..r4
      if (obj.rl1 != null) state.rl1 = obj.rl1; else if (obj.r1 != null) state.rl1 = obj.r1;
      if (obj.rl2 != null) state.rl2 = obj.rl2; else if (obj.r2 != null) state.rl2 = obj.r2;
      if (obj.rl3 != null) state.rl3 = obj.rl3; else if (obj.r3 != null) state.rl3 = obj.r3;
      if (obj.ip) state.ip = obj.ip;
      state.lastAt = Date.now();
      renderTelemetry();
      return;
    } catch (_) {
      if (mode === 'json') { showParsed('Bukan JSON valid'); return; }
    }
  }

  // 2) Coba KV angka
  if (mode === 'kv' || mode === 'auto') {
    const kv = parseKVNumbers(trimmed);
    if (kv) {
      showParsed(kv);
      if (kv.t1 != null) state.t1 = kv.t1;
      if (kv.t2 != null) state.t2 = kv.t2;
      if (kv.t3 != null) state.t3 = kv.t3;
      if (kv.r1 != null) state.rl1 = kv.r1;
      if (kv.r2 != null) state.rl2 = kv.r2;
      if (kv.r3 != null) state.rl3 = kv.r3;
      // mode dari KV string (RUN/DEF)
      const mm = trimmed.match(/\bmode\s*[:=]\s*(RUN|DEF)/i);
      if (mm) {
        const nm = mm[1].toUpperCase() === 'RUN' ? 'RUN' : 'DEF';
        updateMode(nm);
      }
      state.lastAt = Date.now();
      renderTelemetry();
      // lanjutkan ke parseSpecialLogs juga (untuk mode/RL)
    }
  }

  // 3) Pola khusus dari .ino
  parseSpecialLogs(trimmed);
  // setiap baris yang terparse menandakan ESP32 aktif
  state.lastAt = Date.now();
  renderTelemetry();

  // 4) Fallback: raw
  if (mode === 'raw') showParsed(trimmed);
}

async function connectXMPP() {
  try {
    if (!window.XMPP) {
      alert('Pustaka XMPP (StanzaJS) belum dimuat. Pastikan koneksi internet dan coba lagi.');
      return;
    }
    const jid = ui.xmppJid.value.trim();
    const password = ui.xmppPassword.value;
    const wsURL = ui.xmppWsUrl.value.trim();
    if (!jid || !password || !wsURL) {
      alert('Isi JID, Password, dan WS URL terlebih dahulu.');
      return;
    }
    setXMPPStatus('menghubungkan...');
    const client = window.XMPP.createClient({
      jid,
      password,
      transports: { websocket: wsURL },
    });
    client.on('session:started', async () => {
      setXMPPStatus('terhubung');
      ui.xmppConnectBtn.disabled = true;
      ui.xmppDisconnectBtn.disabled = false;
      ui.xmppSendTelemetryBtn.disabled = false;
    });
    client.on('disconnected', () => setXMPPStatus('terputus'));
    client.on('auth:failed', () => setXMPPStatus('auth gagal'));
    client.on('stream:error', (err) => setXMPPStatus(`kesalahan stream: ${err?.condition || err}`));
    await client.start();
    xmppClient = client;
  } catch (err) {
    console.error('connectXMPP error:', err);
    setXMPPStatus(`gagal terhubung: ${err.message || err}`);
  }
}

async function disconnectXMPP() {
  try {
    await xmppClient?.stop?.();
  } catch (err) {
    console.error('disconnectXMPP error:', err);
  } finally {
    setXMPPStatus('terputus');
    ui.xmppConnectBtn.disabled = false;
    ui.xmppDisconnectBtn.disabled = true;
    ui.xmppSendTelemetryBtn.disabled = true;
    xmppClient = null;
  }
}

async function sendTelemetryXMPP() {
  try {
    if (!xmppClient) { alert('XMPP belum terhubung'); return; }
    const to = ui.xmppRecipient.value.trim();
    if (!to) { alert('Isi recipient JID dulu'); return; }
    const payload = {
      type: 'telemetry',
      t1: state.t1, t2: state.t2, t3: state.t3,
      mode: state.mode,
      rl1: state.rl1, rl2: state.rl2, rl3: state.rl3,
      ip: state.ip,
      alarm: state.alarm,
      at: new Date().toISOString(),
    };
    const body = JSON.stringify(payload);
    await xmppClient.sendMessage({ to, body });
    setXMPPStatus('telemetri terkirim');
  } catch (err) {
    console.error('sendTelemetryXMPP error:', err);
    setXMPPStatus(`gagal kirim: ${err.message || err}`);
  }
}

async function readLoop() {
  const textDecoder = new TextDecoderStream();
  port.readable.pipeTo(textDecoder.writable).catch(() => {});
  reader = textDecoder.readable.getReader();
  try {
    while (keepReading) {
      const { value, done } = await reader.read();
      if (done) break;
      if (value) {
        readBuffer += value;
        const lines = readBuffer.split(/\r?\n/);
        readBuffer = lines.pop() ?? '';
        for (const ln of lines) handleLine(ln);
      }
    }
  } catch (err) {
    console.error('readLoop error:', err);
    setStatus(`kesalahan membaca: ${err.message || err}`);
  } finally {
    reader?.releaseLock?.();
  }
}

async function connectSerial() {
  if (!('serial' in navigator)) {
    setStatus('Browser tidak mendukung Web Serial');
    alert('Gunakan Chrome/Edge terbaru.');
    return;
  }
  try {
    const baudRate = Number(ui.baudRate?.value ?? 115200);
    port = await navigator.serial.requestPort();
    await port.open({ baudRate });
    setStatus(`terhubung @ ${baudRate} baud`);
    if (ui.connectBtn) ui.connectBtn.disabled = true;
    if (ui.disconnectBtn) ui.disconnectBtn.disabled = false;
    if (ui.sendBtn) ui.sendBtn.disabled = false;
    ui.defrostBtn.disabled = false;
    writer = port.writable?.getWriter?.();
    keepReading = true;
    readLoop();
  } catch (err) {
    console.error('connectSerial error:', err);
    setStatus(`gagal terhubung: ${err.message || err}`);
  }
}

async function disconnectSerial() {
  try {
    keepReading = false;
    await reader?.cancel?.();
    reader?.releaseLock?.();
    try { writer?.releaseLock?.(); } catch (_) {}
    await port?.close?.();
  } catch (err) {
    console.error('disconnectSerial error:', err);
  } finally {
    setStatus('terputus');
    if (ui.connectBtn) ui.connectBtn.disabled = false;
    if (ui.disconnectBtn) ui.disconnectBtn.disabled = true;
    if (ui.sendBtn) ui.sendBtn.disabled = true;
    if (ui.defrostBtn) ui.defrostBtn.disabled = true;
  }
}

async function sendLine(val) {
  const msg = typeof val === 'string' ? val : ui.sendInput.value;
  if (!msg) return;
  try {
    const encoder = new TextEncoder();
    const data = encoder.encode(msg + '\n');
    if (!writer && port?.writable) writer = port.writable.getWriter();
    await writer.write(data);
    appendLog(`> ${msg}`);
  } catch (err) {
    console.error('sendLine error:', err);
    setStatus(`gagal kirim: ${err.message || err}`);
  }
}

function downloadLog() {
  const blob = new Blob([logs.join('\n')], { type: 'text/plain' });
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  const ts = new Date().toISOString().replace(/[:.]/g, '-');
  a.href = url;
  a.download = `serial-log-${ts}.txt`;
  document.body.appendChild(a);
  a.click();
  a.remove();
  URL.revokeObjectURL(url);
}

function init() {
  if (ui.connectBtn) ui.connectBtn.addEventListener('click', connectSerial);
  if (ui.disconnectBtn) ui.disconnectBtn.addEventListener('click', disconnectSerial);
  if (ui.sendBtn) ui.sendBtn.addEventListener('click', () => sendLine());
  // Pastikan tombol DEFROST aktif secara default (manual via HTTP jika tanpa Serial)
  if (ui.defrostBtn) ui.defrostBtn.disabled = false;
  ui.defrostBtn.addEventListener('click', () => {
    if (state.mode === 'DEF') return; // sudah defrost, abaikan
    // Tampilkan modal konfirmasi tekan-tahan 2 detik
    ui.defrostConfirmModal?.classList.add('show');
  });
  if (ui.downloadBtn) ui.downloadBtn.addEventListener('click', downloadLog);
  ui.applyLimitsBtn?.addEventListener('click', () => {
    const rRaw = ui.limitR?.value ?? '';
    const eRaw = ui.limitE?.value ?? '';
    const oRaw = ui.limitO?.value ?? '';

    const rNorm = rRaw === '' ? null : normalizeNumber(rRaw);
    const eNorm = eRaw === '' ? null : normalizeNumber(eRaw);
    const oNorm = oRaw === '' ? null : normalizeNumber(oRaw);

    const r = rNorm == null ? null : Number(rNorm.toFixed(1));
    const e = eNorm == null ? null : Number(eNorm.toFixed(1));
    const o = oNorm == null ? null : Number(oNorm.toFixed(1));

    // Tampilkan modal konfirmasi
    if (ui.applyConfirmR) ui.applyConfirmR.textContent = (r == null ? '(kosong)' : r.toFixed(1));
    if (ui.applyConfirmE) ui.applyConfirmE.textContent = (e == null ? '(kosong)' : e.toFixed(1));
    if (ui.applyConfirmO) ui.applyConfirmO.textContent = (o == null ? '(kosong)' : o.toFixed(1));
    if (ui.applyConfirmModal) ui.applyConfirmModal.classList.add('show');

    // Simpan nilai sementara untuk aksi konfirmasi
    ui.applyConfirmYes?.setAttribute('data-r', r == null ? '' : String(r));
    ui.applyConfirmYes?.setAttribute('data-e', e == null ? '' : String(e));
    ui.applyConfirmYes?.setAttribute('data-o', o == null ? '' : String(o));
  });

  // Modal: batal dan konfirmasi
  ui.applyConfirmNo?.addEventListener('click', () => {
    ui.applyConfirmModal?.classList.remove('show');
    appendLog('> Terapkan batas dibatalkan');
  });
  // Modal DEFROS: batal
  ui.defrostConfirmNo?.addEventListener('click', () => {
    ui.defrostConfirmModal?.classList.remove('show');
    appendLog('> Manual DEFROS dibatalkan');
  });
  // Press-and-hold 2 detik untuk konfirmasi
  const HOLD_MS = 2000;
  let holdTimer = null;
  let holdRaf = null;
  let holdStart = 0;
  const cancelHold = () => {
    if (holdTimer) { clearTimeout(holdTimer); holdTimer = null; }
    if (holdRaf) { cancelAnimationFrame(holdRaf); holdRaf = null; }
    holdStart = 0;
    if (ui.applyConfirmProgress) ui.applyConfirmProgress.style.width = '0%';
    ui.applyConfirmYes?.classList.remove('holding');
  };
  const startHold = (ev) => {
    ev.preventDefault();
    if (!ui.applyConfirmYes) return;
    ui.applyConfirmYes.classList.add('holding');
    holdStart = performance.now();
    const update = (ts) => {
      const elapsed = ts - holdStart;
      const pct = Math.max(0, Math.min(100, (elapsed / HOLD_MS) * 100));
      if (ui.applyConfirmProgress) ui.applyConfirmProgress.style.width = pct + '%';
      if (elapsed >= HOLD_MS) return; else holdRaf = requestAnimationFrame(update);
    };
    holdRaf = requestAnimationFrame(update);
    holdTimer = setTimeout(async () => {
      // selesai hold → konfirmasi
      cancelHold();
      const rAttr = ui.applyConfirmYes?.getAttribute('data-r');
      const eAttr = ui.applyConfirmYes?.getAttribute('data-e');
      const oAttr = ui.applyConfirmYes?.getAttribute('data-o');
      const r = rAttr === '' || rAttr == null ? null : Number(rAttr);
      const e = eAttr === '' || eAttr == null ? null : Number(eAttr);
      const o = oAttr === '' || oAttr == null ? null : Number(oAttr);
      ui.applyConfirmModal?.classList.remove('show');

      state.limits = { R: r, E: e, O: o };
      if (ui.limitR) ui.limitR.value = r == null ? '' : r.toFixed(1);
      if (ui.limitE) ui.limitE.value = e == null ? '' : e.toFixed(1);
      if (ui.limitO) ui.limitO.value = o == null ? '' : o.toFixed(1);
      renderTelemetry();
      try {
        ui.applyLimitsBtn && (ui.applyLimitsBtn.disabled = true);
        await saveLimitsToBackend(r, e, o);
        appendLog('> Batas telemetri tersimpan');
      } catch (err) {
        alert('Gagal menyimpan batas ke backend: ' + (err.message || err));
      } finally {
        ui.applyLimitsBtn && (ui.applyLimitsBtn.disabled = false);
      }
    }, HOLD_MS);
  };
  // Gunakan pointer events, fallback ke mouse, dan dukungan touch
  if (ui.applyConfirmYes) {
    // Pointer
    ui.applyConfirmYes.addEventListener('pointerdown', startHold);
    ui.applyConfirmYes.addEventListener('pointerup', cancelHold);
    ui.applyConfirmYes.addEventListener('pointerleave', cancelHold);
    ui.applyConfirmYes.addEventListener('pointercancel', cancelHold);
    // Mouse
    ui.applyConfirmYes.addEventListener('mousedown', startHold);
    ui.applyConfirmYes.addEventListener('mouseup', cancelHold);
    ui.applyConfirmYes.addEventListener('mouseleave', cancelHold);
    // Touch (untuk browser tanpa Pointer Events)
    ui.applyConfirmYes.addEventListener('touchstart', (e) => { e.preventDefault(); startHold(e); }, { passive: false });
    ui.applyConfirmYes.addEventListener('touchend', (e) => { e.preventDefault(); cancelHold(e); }, { passive: false });
    ui.applyConfirmYes.addEventListener('touchcancel', (e) => { e.preventDefault(); cancelHold(e); }, { passive: false });
  }

  // Hold-to-confirm untuk DEFROS (2 detik)
  const DEF_HOLD_MS = 2000;
  let defHoldTimer = null;
  let defHoldRaf = null;
  let defHoldStart = 0;
  const defCancelHold = () => {
    if (defHoldTimer) { clearTimeout(defHoldTimer); defHoldTimer = null; }
    if (defHoldRaf) { cancelAnimationFrame(defHoldRaf); defHoldRaf = null; }
    defHoldStart = 0;
    if (ui.defrostConfirmProgress) ui.defrostConfirmProgress.style.width = '0%';
    ui.defrostConfirmYes?.classList.remove('holding');
  };
  const defStartHold = (ev) => {
    ev.preventDefault();
    if (!ui.defrostConfirmYes) return;
    ui.defrostConfirmYes.classList.add('holding');
    defHoldStart = performance.now();
    const update = (ts) => {
      const elapsed = ts - defHoldStart;
      const pct = Math.max(0, Math.min(100, (elapsed / DEF_HOLD_MS) * 100));
      if (ui.defrostConfirmProgress) ui.defrostConfirmProgress.style.width = pct + '%';
      if (elapsed >= DEF_HOLD_MS) return; else defHoldRaf = requestAnimationFrame(update);
    };
    defHoldRaf = requestAnimationFrame(update);
    defHoldTimer = setTimeout(async () => {
      // selesai hold → jalankan DEFROS
      defCancelHold();
      ui.defrostConfirmModal?.classList.remove('show');
      // Jika Serial terhubung, kirim perintah langsung.
      if (port) {
        sendLine('DEFROST');
        return;
      }
      // Jika tidak, trigger lewat backend HTTP agar ESP32 mendeteksi via polling state.php
      try {
        const res = await fetch(BACKEND_BASE_URL + 'state.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
          body: 'state=1',
          cache: 'no-store',
        });
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        await res.json();
        appendLog('> Manual DEFROS dikirim');
      } catch (errAbs) {
        console.warn('Gagal trigger ke backend absolut, coba lokal:', errAbs);
        try {
          const res = await fetch('mulaibaru/backend/state.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: 'state=1',
            cache: 'no-store',
          });
          if (!res.ok) throw new Error(`HTTP ${res.status}`);
          await res.json();
          appendLog('> Manual DEFROS dikirim');
        } catch (errLoc) {
          alert('Gagal trigger DEFROST via HTTP (lokal): ' + (errLoc.message || errLoc));
        }
      }
    }, DEF_HOLD_MS);
  };
  // Event binding untuk tombol konfirmasi DEFROS
  if (ui.defrostConfirmYes) {
    // Pointer
    ui.defrostConfirmYes.addEventListener('pointerdown', defStartHold);
    ui.defrostConfirmYes.addEventListener('pointerup', defCancelHold);
    ui.defrostConfirmYes.addEventListener('pointerleave', defCancelHold);
    ui.defrostConfirmYes.addEventListener('pointercancel', defCancelHold);
    // Mouse
    ui.defrostConfirmYes.addEventListener('mousedown', defStartHold);
    ui.defrostConfirmYes.addEventListener('mouseup', defCancelHold);
    ui.defrostConfirmYes.addEventListener('mouseleave', defCancelHold);
    // Touch
    ui.defrostConfirmYes.addEventListener('touchstart', (e) => { e.preventDefault(); defStartHold(e); }, { passive: false });
    ui.defrostConfirmYes.addEventListener('touchend', (e) => { e.preventDefault(); defCancelHold(e); }, { passive: false });
    ui.defrostConfirmYes.addEventListener('touchcancel', (e) => { e.preventDefault(); defCancelHold(e); }, { passive: false });
  }
  // XMPP
  ui.xmppConnectBtn?.addEventListener('click', connectXMPP);
  ui.xmppDisconnectBtn?.addEventListener('click', disconnectXMPP);
  ui.xmppSendTelemetryBtn?.addEventListener('click', sendTelemetryXMPP);
  setStatus('belum terhubung');
  // Pulihkan mode dari localStorage sebelum polling backend
  loadModeFromStorage();
  updateModeDurationUI();
  // Muat batas tersimpan dari backend saat inisialisasi
  loadLimitsFromBackend();
  // Sinkronisasi limits antar klien: polling berkala
  setInterval(pollLimitsFromBackend, 2000);

  // Stepper binding untuk tombol +/− di pengaturan telemetri
  document.querySelectorAll('.stepper .dec').forEach(btn => {
    btn.addEventListener('click', () => {
      const target = btn.dataset.target;
      adjustInput(target, -1);
    });
  });
  document.querySelectorAll('.stepper .inc').forEach(btn => {
    btn.addEventListener('click', () => {
      const target = btn.dataset.target;
      adjustInput(target, +1);
    });
  });

  // Polling backend sebagai fallback jika Web Serial belum terhubung
  async function pollBackendLast() {
    if (port) return;
    // Helper timeout untuk fetch
    const fetchWithTimeout = async (url, opts, timeoutMs) => {
      const controller = new AbortController();
      const id = setTimeout(() => controller.abort(), timeoutMs);
      try {
        const res = await fetch(url, { ...(opts||{}), signal: controller.signal });
        return res;
      } finally {
        clearTimeout(id);
      }
    };
    // 1) Coba backend absolut dengan timeout singkat
    try {
      const resAbs = await fetchWithTimeout(BACKEND_BASE_URL + 'receive.php?last=1', { cache: 'no-store' }, 2000);
      if (resAbs.ok) {
        const obj = await resAbs.json();
        const n1 = normalizeNumber(obj.t1);
        const n2 = normalizeNumber(obj.t2);
        const n3 = normalizeNumber(obj.t3);
        if (n1 != null) state.t1 = n1;
        if (n2 != null) state.t2 = n2;
        if (n3 != null) state.t3 = n3;
        if (obj.mode) updateMode(obj.mode);
        if (obj.dur != null && Number.isFinite(Number(obj.dur))) {
          const ds = Number(obj.dur);
          if (ds >= 0) state.modeChangedAt = Date.now() - ds * 1000;
          saveModeToStorage();
        }
        if (obj.ip) state.ip = obj.ip;
        ['rl1','rl2','rl3'].forEach(k => { if (obj[k] != null) state[k] = obj[k]; });
        try { state.lastAt = obj.at ? (new Date(obj.at)).getTime() : Date.now(); } catch (_) { state.lastAt = Date.now(); }
        renderTelemetry();
        return;
      }
    } catch (errAbs) {
      console.warn('Gagal fetch last dari backend absolut (timeout singkat):', errAbs);
    }
    // 2) Fallback ke proxy lokal dengan timeout singkat
    try {
      const resProxy = await fetchWithTimeout('mulaibaru/backend/receive_proxy.php?last=1', { cache: 'no-store' }, 2000);
      if (resProxy.ok) {
        const obj = await resProxy.json();
        const n1 = normalizeNumber(obj.t1);
        const n2 = normalizeNumber(obj.t2);
        const n3 = normalizeNumber(obj.t3);
        if (n1 != null) state.t1 = n1;
        if (n2 != null) state.t2 = n2;
        if (n3 != null) state.t3 = n3;
        if (obj.mode) updateMode(obj.mode);
        if (obj.dur != null && Number.isFinite(Number(obj.dur))) {
          const ds = Number(obj.dur);
          if (ds >= 0) state.modeChangedAt = Date.now() - ds * 1000;
          saveModeToStorage();
        }
        if (obj.ip) state.ip = obj.ip;
        ['rl1','rl2','rl3'].forEach(k => { if (obj[k] != null) state[k] = obj[k]; });
        try { state.lastAt = obj.at ? (new Date(obj.at)).getTime() : Date.now(); } catch (_) { state.lastAt = Date.now(); }
        renderTelemetry();
        return;
      }
    } catch (errProxy) {
      console.warn('Gagal fetch via proxy lokal (timeout singkat):', errProxy);
    }
    // 3) Tampilkan dari file lokal jika ada
    try {
      const resLoc = await fetch('mulaibaru/backend/data/last.json', { cache: 'no-store' });
      if (resLoc.ok) {
        const obj = await resLoc.json();
        const nn1 = normalizeNumber(obj.t1);
        const nn2 = normalizeNumber(obj.t2);
        const nn3 = normalizeNumber(obj.t3);
        if (nn1 != null) state.t1 = nn1;
        if (nn2 != null) state.t2 = nn2;
        if (nn3 != null) state.t3 = nn3;
        if (obj.mode) updateMode(obj.mode);
        if (obj.dur != null && Number.isFinite(Number(obj.dur))) {
          const ds = Number(obj.dur);
          if (ds >= 0) state.modeChangedAt = Date.now() - ds * 1000;
          saveModeToStorage();
        }
        if (obj.ip) state.ip = obj.ip;
        ['rl1','rl2','rl3'].forEach(k => { if (obj[k] != null) state[k] = obj[k]; });
        try { state.lastAt = obj.at ? (new Date(obj.at)).getTime() : Date.now(); } catch (_) { state.lastAt = Date.now(); }
        renderTelemetry();
      }
    } catch (_e) {}
  }
  pollBackendLast();
  setInterval(pollBackendLast, 2000);
  // Tick setiap detik untuk memperbarui durasi mode di UI
  setInterval(updateModeDurationUI, 1000);

  // Muat histori saat awal dan segarkan setiap menit
  loadHistory();
  ui.historyRefreshBtn?.addEventListener('click', loadHistory);
}

window.addEventListener('DOMContentLoaded', init);
