#include <WiFi.h>
#include <HTTPClient.h>
#include <Ticker.h>
//#include <EEPROM.h>
#include <LiquidCrystal_I2C.h>
#include <math.h>
#include <Preferences.h>
// ------------------ Konfigurasi WiFi & Server ------------------
const char* WIFI_SSID = "Jekson";
const char* WIFI_PASS = "1234567890";
 const char* SERVER_URL = "http://afikaservice.my.id/mulai/mulaibaru/backend/";
const char* API_KEY    =  "b2d7a1f0e9c44bb78a9031e5c6d2f41a7e3d9ac1";

// ------------------ Pin Definisi ------------------
#define NTC_PIN1 34
#define NTC_PIN2 35
#define NTC_PIN3 32
int selectedLimit = 0;
int lastSelectedLimit = -1;
const int Rl1 = 5;   // kompresor
const int Rl2 = 17;  // pemanas defrost
const int Rl3 = 16;  // kontrol temp1 (evap)
const int GPIO2_PIN = 2; // DEFROS pulse 2s (dari web)
const int defrostButtonPin = 15;
const int buzzerPin = 27;  // ubah sesuai pin buzzer yang kamu pakai

#define BTN_UP    19
#define BTN_DOWN  13
#define BTN_NEXT  18   // untuk pilih setpoint mana yang mau diubah
Preferences prefs;
float setR, setE, setO;
int currentSet = 0; // 0=Ruang, 1=Evap, 2=Outdoor
// ------------------ Konstanta NTC ------------------
#define SERIES_RESISTOR      10000.0
#define NOMINAL_RESISTANCE   10000.0
#define NOMINAL_TEMPERATURE  25.0
#define BETA_COEFFICIENT     3950.0
#define ADC_MAX              4095.0
#define VOLTAGE_REF          3.3

// ------------------ Waktu via Ticker ------------------
Ticker ticker;                    
volatile int detik=0, menit=0, jam=0;

// ------------------ Status & EEPROM ------------------
enum SystemState { RUNNING, DEFROST };
SystemState currentState = RUNNING;

// Relay state (global agar bisa di-reset dari fungsi lain)
bool rl1State = false;
bool rl3State = false;

// ------------------ LCD ------------------
LiquidCrystal_I2C lcd(0x27, 20, 4);
bool blinkState = false;
unsigned long lastBlinkTime = 0;
const unsigned long BLINK_INTERVAL = 500; // 0.5 detik per kedipan

// cache tampilan
float lastTemp1 = -999.0, lastTemp2 = -999.0, lastTemp3 = -999.0;
SystemState lastState = DEFROST;
int lastJam=-1, lastMenit=-1, lastDetik=-1;

// ------------------ Delay ON Relay (ms) ------------------
const unsigned long RELAY_ON_DELAY = 10000; // 10 detik
unsigned long rl1PendingTime = 0, rl3PendingTime = 0;
bool rl1Pending=false, rl3Pending=false;

// ------------------ Web Sched ------------------
unsigned long lastWebPost   = 0;
const unsigned long WEB_POST_INTERVAL = 5000;
unsigned long lastWebPoll   = 0;
const unsigned long WEB_POLL_INTERVAL = 1000;

// ------------------ DEFROS ------------------
unsigned long defrozStart = 0;
bool defrozActive = false;
// ------------------ Alarm suhu ------------------
unsigned long alarmStartTime = 0;
bool alarmActive = false;

// ------------------ Tombol / Setpoint ------------------
unsigned long lastDefrostPress = 0;
unsigned long lastButtonPress = 0;
const unsigned long debounceDelay = 200; // ms

// ------------------ WiFi ------------------
unsigned long lastWifiCheck = 0;
const unsigned long WIFI_CHECK_INTERVAL = 5000;
// Sinkronisasi batas dari backend
unsigned long lastLimitsSync = 0;
const unsigned long LIMITS_SYNC_INTERVAL = 60000; // 60 detik
// --- Forward Declaration ---
float readTemperature(int pin);
void displayDataOnLcd(float temp1, float temp2, float temp3);
void postTemperatures(float t1, float t2, float t3);
void pollAndHandleDefros();
void updateGpio2AutoOffAndResetServer();
void handleSystemState();
void handleRelays(float temp1, float temp2, float temp3);
void checkButtons();
void resetRelayLogic(bool fromManual = false);
void restoreRelaysAfterDefrost(float t1, float t2, float t3);
void syncLimitsFromBackend();


// ================== SETUP ==================
void setup() {
  Serial.begin(115200);

  // IO
  pinMode(Rl1, OUTPUT);
  pinMode(Rl2, OUTPUT);
  pinMode(Rl3, OUTPUT);
  pinMode(GPIO2_PIN, OUTPUT);
  pinMode(defrostButtonPin, INPUT_PULLUP);
  pinMode(BTN_UP, INPUT_PULLUP);
  pinMode(BTN_DOWN, INPUT_PULLUP);
  pinMode(BTN_NEXT, INPUT_PULLUP);
  pinMode(buzzerPin, OUTPUT);


  // Preferences
  prefs.begin("setpoints", false);
  setR = prefs.getFloat("setR", 25.0);
  setE = prefs.getFloat("setE", -5.0);
  setO = prefs.getFloat("setO", 30.0);

  // Relay awal OFF (LOW = OFF, HIGH = ON)
digitalWrite(Rl1, LOW);
digitalWrite(Rl2, LOW);
digitalWrite(Rl3, LOW);
  digitalWrite(GPIO2_PIN, HIGH);
digitalWrite(buzzerPin, LOW);
  // sinkron internal states
  rl1State = rl3State = false;
  rl1Pending = rl3Pending = false;
  rl1PendingTime = rl3PendingTime = 0;

  // LCD
  lcd.init();
  lcd.backlight();
  lcd.clear();
  lcd.setCursor(0,0); lcd.print("SIYAU"); delay(1000);
  lcd.clear();
  lcd.setCursor(0,1); lcd.print("BELIAU SEKAWAN"); delay(1200);
  lcd.clear();

  // WiFi
  WiFi.persistent(true);
  connectToWiFi();

  // Ticker 1Hz
  ticker.attach(1, isrFunc);

  Serial.println("Setup selesai.");
}

float readTemperature(int pin){
  float adcValue = analogRead(pin);
  if(adcValue <= 0.5) return -273.0;
  float voltage = adcValue * VOLTAGE_REF / ADC_MAX;
  float resistance = SERIES_RESISTOR * (VOLTAGE_REF / voltage - 1.0);

  float steinhart = resistance / NOMINAL_RESISTANCE;
  steinhart = log(steinhart);
  steinhart /= BETA_COEFFICIENT;
  steinhart += 1.0 / (NOMINAL_TEMPERATURE + 273.15);
  steinhart = 1.0 / steinhart;
  steinhart -= 273.15;
  return steinhart;
}

// ================== LOOP ==================
void loop() {
  // WiFi reconnect
  if(millis() - lastWifiCheck > WIFI_CHECK_INTERVAL){
    lastWifiCheck = millis();
    checkAndReconnectWiFi();
  }
  // Baca perintah dari Serial (contoh: DEFROST)
  handleSerialCommands();
 
  // Baca suhu
  float t1 = readTemperature(NTC_PIN1);
  float t2 = readTemperature(NTC_PIN2);
  float t3 = readTemperature(NTC_PIN3);
  // ====== Cek Alarm Suhu Ruang (T1) ======
// ====== Cek Alarm Suhu Ruang (T1) ======
// Hanya aktif jika sistem TIDAK dalam mode DEFROST
if (!defrozActive) {
  if (t1 < 0 || t1 > 10) {
    if (alarmStartTime == 0) {
      alarmStartTime = millis();
    } else if ((millis() - alarmStartTime) >= 600000 && !alarmActive) { // 10 menit
      digitalWrite(buzzerPin, HIGH);
      alarmActive = true;
      Serial.println("⚠️ ALARM AKTIF: Suhu T1 di luar batas selama >10 menit");
    }
  } else {
    alarmStartTime = 0;
    if (alarmActive) {
      digitalWrite(buzzerPin, LOW);
      alarmActive = false;
      Serial.println("✅ Suhu normal, buzzer OFF");
    }
  }
} else {
  // Jika sedang defrost, matikan alarm & reset timer
  alarmStartTime = 0;
  if (alarmActive) {
    digitalWrite(buzzerPin, LOW);
    alarmActive = false;
    Serial.println("🧊 DEFROST MODE: Alarm dinonaktifkan");
  }
}

  // Tombol
  checkButtons();

  // Mode RUNNING/DEFROST
  handleSystemState();

  // Relay
  handleRelays(t1, t2, t3);

  // LCD
  displayDataOnLcd(t1, t2, t3);

  // Web POST suhu
  if (millis() - lastWebPost >= WEB_POST_INTERVAL) {
    lastWebPost = millis();
    postTemperatures(t1, t2, t3);
  }

  // Poll server untuk trigger defrost
  if (millis() - lastWebPoll >= WEB_POLL_INTERVAL) {
    lastWebPoll = millis();
    pollAndHandleDefros();
  }

  // Sync limits (setR/E/O) dari backend ke Preferences
  if (millis() - lastLimitsSync >= LIMITS_SYNC_INTERVAL) {
    lastLimitsSync = millis();
    syncLimitsFromBackend();
  }

  // Auto OFF GPIO2 (pulse)
  updateGpio2AutoOffAndResetServer();

  delay(10);
}

void tampilSetpoint(){
  lcd.clear();
  lcd.setCursor(0,20);
  lcd.print("Setpoint:");
}

// ================== IMPLEMENTASI ==================
void isrFunc() {
  detik++;
  if (detik >= 60) { detik = 0; menit++; }
  if (menit >= 60) { menit = 0; jam++; if (jam >= 24) jam = 0; }
}

void connectToWiFi() {
  Serial.printf("Menghubungkan ke WiFi SSID: %s\n", WIFI_SSID);
  WiFi.begin(WIFI_SSID, WIFI_PASS);
  unsigned long start = millis();
  while(WiFi.status() != WL_CONNECTED && millis() - start < 10000){
    delay(500); Serial.print(".");
  }
  if(WiFi.status() == WL_CONNECTED){
    Serial.printf("\nWiFi terhubung, IP: %s\n", WiFi.localIP().toString().c_str());
  }else{
    Serial.println("\nGagal konek WiFi.");
  }
}

void checkAndReconnectWiFi(){
  if(WiFi.status() != WL_CONNECTED){
    Serial.println("WiFi putus, reconnect...");
    WiFi.disconnect();
    WiFi.begin(WIFI_SSID, WIFI_PASS);
  }
}

// ------------------ Tombol ------------------
void checkButtons(){
  // DEFROST manual (GPIO15)
  if (digitalRead(defrostButtonPin) == LOW && (millis() - lastDefrostPress > debounceDelay)) {
    if(currentState == RUNNING){
      currentState = DEFROST;
      noInterrupts(); jam=0; menit=0; detik=0; interrupts();
      Serial.println("DEFROST manual dari tombol fisik");

      // reset logic & mulai defrost
      resetRelayLogic(true);        // fromManual = true
      digitalWrite(Rl2, HIGH);      // heater ON
      
      defrozStart = millis();
      defrozActive = true;
    }
    lastDefrostPress = millis();
  }

  // SET / UP / DOWN
  if (millis() - lastButtonPress < debounceDelay) return;

  if (digitalRead(BTN_UP) == LOW) {
    if (selectedLimit == 0) setR++;
    else if (selectedLimit == 1) setE++;
    else if (selectedLimit == 2) setO++;
    saveSetpoints();
    lastButtonPress = millis();
  }

  if (digitalRead(BTN_DOWN) == LOW) {
    if (selectedLimit == 0) setR--;
    else if (selectedLimit == 1) setE--;
    else if (selectedLimit == 2) setO--;
    saveSetpoints();
    lastButtonPress = millis();
  }
  if (digitalRead(BTN_NEXT) == LOW) {
    selectedLimit++;
    if (selectedLimit > 2) selectedLimit = 0;  // balik ke Ruang lagi
    lastButtonPress = millis();
  }
}

// ====== SIMPAN KE FLASH ======
void saveSetpoints() {
  prefs.putFloat("setR", setR);
  prefs.putFloat("setE", setE);
  prefs.putFloat("setO", setO);
}

// ------------------ Mode RUN/DEFROST ------------------
void resetRelayLogic(bool fromManual) {
  // Reset semua status dan timer pending relay
  rl1Pending = rl3Pending = false;
  rl1PendingTime = rl3PendingTime = 0;

  // Matikan semua relay pending (kompresor dan kontrol suhu)
  digitalWrite(Rl1, LOW);
  digitalWrite(Rl3, LOW);

  // Reset state internal supaya logika sinkron dengan keadaan fisik
  rl1State = false;
  rl3State = false;

  if(fromManual) Serial.println("resetRelayLogic(): dipanggil dari MANUAL defrost");
  else Serial.println("resetRelayLogic(): dipanggil dari AUTO/web defrost");
}

// fungsi bantu: setelah defrost selesai, set pending (JANGAN langsung ON)
void restoreRelaysAfterDefrost(float t1, float t2, float t3){
  unsigned long now = millis();

  if (t1 >= setR + 1 && t2 >= setE + 1) {
    rl1Pending = true;
    rl1PendingTime = now;
    Serial.println("restoreRelaysAfterDefrost: set pending RL1 (kompresor)");
  } else {
    rl1Pending = false;
  }

  if (t1 >= setR + 1 && t3 >= setO + 1) {
    rl3Pending = true;
    rl3PendingTime = now;
    Serial.println("restoreRelaysAfterDefrost: set pending RL3 (evap)");
  } else {
    rl3Pending = false;
  }
}

void handleSystemState() {
  if (currentState == RUNNING) {
    digitalWrite(Rl2, LOW); // heater defrost OFF

    // Auto masuk defrost tiap 3 jam
    if (jam >= 3) {
      currentState = DEFROST;
      noInterrupts(); jam = 0; menit = 0; detik = 0; interrupts();
      resetRelayLogic(false); // matikan semua relay sebelum defrost (auto)
      digitalWrite(Rl2, HIGH); // heater ON
      
      defrozStart = millis();
      defrozActive = true;
      Serial.println("Masuk mode DEFROST otomatis.");
    }

  } else if (currentState == DEFROST) {
    // Heater ON, kompresor OFF (pastikan)
    digitalWrite(Rl1, LOW);
    digitalWrite(Rl2, HIGH);

    // Setelah 10 menit, kembali ke RUNNING
    if (menit >= 10) {
      // baca suhu terbaru sebelum restore (agar pending sesuai kondisi nyata)
      float t1 = readTemperature(NTC_PIN1);
      float t2 = readTemperature(NTC_PIN2);
      float t3 = readTemperature(NTC_PIN3);

      currentState = RUNNING;
      noInterrupts(); jam = 0; menit = 0; detik = 0; interrupts();

      resetRelayLogic(false); // reset pending agar kompresor bisa aktif lagi
      digitalWrite(Rl2, HIGH); // heater tetap ON; akan dimatikan saat kembali RUNNING
      defrozActive = false;
      Serial.println("DEFROST selesai → kembali ke RUNNING.");

      // set pending (tidak langsung ON)
      restoreRelaysAfterDefrost(t1, t2, t3);
    }
  }
}

// ------------------ Relay Control ------------------
void handleRelays(float temp1,float temp2,float temp3){
  unsigned long now=millis();

  if (!defrozActive) {
    // RL1 (kompresor)
    if(!rl1State && (temp1>=setR+1 && temp2>=setE+1)){
      if(!rl1Pending){ rl1PendingTime=now; rl1Pending=true; Serial.println("handleRelays: start pending RL1"); }
      else if(now-rl1PendingTime>=RELAY_ON_DELAY){
        digitalWrite(Rl1,HIGH);
        rl1State=true;
        rl1Pending=false;
        Serial.println("RL1 -> ON (kompresor) dari handleRelays");
      }
    } else if(rl1State && (temp1<=setR || temp2<=setE)){
      digitalWrite(Rl1,LOW);
      rl1State=false;
      rl1Pending=false;
      Serial.println("RL1 -> OFF (kompresor) dari handleRelays");
    } else if(!rl1State){
      rl1Pending=false;
    }

    // RL3 (evap)
    if(!rl3State && (temp1>=setR+1 && temp3>=setO+1)){
      if(!rl3Pending){ rl3PendingTime=now; rl3Pending=true; Serial.println("handleRelays: start pending RL3"); }
      else if(now-rl3PendingTime>=RELAY_ON_DELAY){
        digitalWrite(Rl3,HIGH);
        rl3State=true;
        rl3Pending=false;
        Serial.println("RL3 -> ON (evap) dari handleRelays");
      }
    } else if(rl3State && (temp1<=setR || temp3<=setO)){
      digitalWrite(Rl3,LOW);
      rl3State=false;
      rl3Pending=false;
      Serial.println("RL3 -> OFF (evap) dari handleRelays");
    } else if(!rl3State){
      rl3Pending=false;
    }

  }

  // Defrost relay (pastikan OFF saat RUNNING)
  if(currentState==RUNNING) digitalWrite(Rl2,LOW);
}

// ------------------ LCD ------------------
void displayDataOnLcd(float temp1, float temp2, float temp3){
  lcd.setCursor(0,0); lcd.print("RUANG"); lcd.setCursor(0,1); lcd.print((int)round(temp1)); lcd.print("C  ");
  lcd.setCursor(6,0); lcd.print("EVAP"); lcd.setCursor(6,1); lcd.print((int)round(temp2)); lcd.print("C  ");
  lcd.setCursor(11,0); lcd.print("OPT"); lcd.setCursor(11,1); lcd.print((int)round(temp3)); lcd.print("C  ");

  lcd.setCursor(0,3); lcd.print("R:"); lcd.print((int)round(setR)); lcd.print("  ");
  lcd.setCursor(6,3); lcd.print("E:"); lcd.print((int)round(setE)); lcd.print("  ");
  lcd.setCursor(12,3); lcd.print("O:"); lcd.print((int)round(setO)); lcd.print("  ");
  lcd.setCursor(17,1); lcd.print(currentState==RUNNING?"RUN":"DEF");
  if(selectedLimit!=lastSelectedLimit){
    lcd.setCursor(19,3);
    if(selectedLimit==0) lcd.print("R");
    else if(selectedLimit==1) lcd.print("E");
    else lcd.print("O");
    lastSelectedLimit=selectedLimit;
  }
  char buf[9]; snprintf(buf,sizeof(buf),"%02d:%02d:%02d",jam,menit,detik);
  lcd.setCursor(0,2); lcd.print(buf);
  lcd.setCursor(11,2); lcd.print("WiFi:"); lcd.print(WiFi.status()==WL_CONNECTED?"OK ":"off");

  int r1 = (digitalRead(Rl1)==HIGH)?1:0;
int r2 = (digitalRead(Rl2)==HIGH)?1:0;
int r3 = (digitalRead(Rl3)==HIGH)?1:0;


  // Logika kedipan 0.5 detik
  if (millis() - lastBlinkTime >= BLINK_INTERVAL) {
    blinkState = !blinkState;
    lastBlinkTime = millis();
  }

  lcd.setCursor(16,0);

  // Rl1 — kompresor
  if (rl1Pending && blinkState) lcd.print(" ");
  else lcd.print(r1);

  // Rl2 — heater defrost (tidak pakai pending)
  lcd.print(r2);

  // Rl3 — evaporator
  if (rl3Pending && blinkState) lcd.print(" ");
  else lcd.print(r3);


  // ——— Serial Telemetry (1 Hz) ———
  // Format yang mudah diparse oleh web: "t1=.. t2=.. t3=.. mode=RUN r1=1 r2=0 r3=1 r4=0"
  static unsigned long lastSer = 0;
  unsigned long nowSer = millis();
  if (nowSer - lastSer >= 1000) {
    lastSer = nowSer;
    const char* modeStrSer = (currentState==RUNNING ? "RUN" : "DEF");
    int durSec = jam*3600 + menit*60 + detik; // durasi mode saat ini dalam detik
    Serial.printf("{\"t1\":%.2f,\"t2\":%.2f,\"t3\":%.2f,\"mode\":\"%s\",\"r1\":%d,\"r2\":%d,\"r3\":%d,\"dur\":%d}\n",
                  temp1, temp2, temp3, modeStrSer, r1, r2, r3, durSec);
  }
}

// ------------------ Web ------------------
void postTemperatures(float t1, float t2, float t3){
  if(WiFi.status()!=WL_CONNECTED) return;
  HTTPClient http;
  String url = String(SERVER_URL) + "receive.php";
  http.begin(url);
  http.addHeader("Content-Type","application/x-www-form-urlencoded");

  String modeStr = currentState==RUNNING?"RUN":"DEF";
  String postData = "api_key="+String(API_KEY)+"&t1="+String(t1,2)+"&t2="+String(t2,2)+"&t3="+String(t3,2)+"&mode="+modeStr;
  // Sertakan status relay berdasarkan pembacaan hardware agar akurat
  int r1hw = (digitalRead(Rl1)==HIGH)?1:0;
  int r2hw = (digitalRead(Rl2)==HIGH)?1:0;
  int r3hw = (digitalRead(Rl3)==HIGH)?1:0;
  // Kirim dua format: rlX (legacy) dan rX (hardware)
  postData += "&rl1=" + String(r1hw);
  postData += "&rl2=" + String(r2hw);
  postData += "&rl3=" + String(r3hw);
  postData += "&r1=" + String(r1hw);
  postData += "&r2=" + String(r2hw);
  postData += "&r3=" + String(r3hw);
  postData += "&ip=" + WiFi.localIP().toString();
  postData += "&alarm=" + String(alarmActive ? 1 : 0);
  // Sertakan durasi mode (detik) agar UI dapat menyelaraskan modeChangedAt
  int durSec = jam*3600 + menit*60 + detik;
  postData += "&dur=" + String(durSec);

  int code = http.POST(postData);
  if(code>0) Serial.printf("[POST receive] %d: %s\n", code, http.getString().c_str());
  http.end();
}

// Trigger Defrost dari Web
void pollAndHandleDefros(){
  if(WiFi.status()!=WL_CONNECTED) return;
  HTTPClient http;
  http.begin(String(SERVER_URL)+"state.php");
  int code = http.GET();
  if(code==200){
    String resp = http.getString();
    if(resp.indexOf("\"gpio2\":1")>=0 && !defrozActive){
      digitalWrite(GPIO2_PIN,LOW);
      defrozStart = millis();
      defrozActive=true;
      currentState = DEFROST;
      noInterrupts(); jam=0; menit=0; detik=0; interrupts();
      Serial.println("DEFROST manual dari WEB");

      // pastikan logic reset saat web-trigger manual defrost
      resetRelayLogic(true);
      digitalWrite(Rl2, HIGH); // heater ON
      
    }
  }
  http.end();
}

// Ambil limits (R/E/O) dari backend/state.php dan simpan ke Preferences jika berubah
void syncLimitsFromBackend(){
  if (WiFi.status() != WL_CONNECTED) return;
  HTTPClient http;
  http.begin(String(SERVER_URL)+"state.php");
  int code = http.GET();
  if (code == 200) {
    String resp = http.getString();
    int pLimits = resp.indexOf("\"limits\"");
    if (pLimits >= 0) {
      float newR = NAN, newE = NAN, newO = NAN;
      // Cari R
      int pR = resp.indexOf("\"R\":", pLimits);
      if (pR >= 0) {
        int s = pR + 4; // setelah "R":
        int e = resp.indexOf(',', s);
        if (e < 0) e = resp.indexOf('}', s);
        if (e > s) {
          String rStr = resp.substring(s, e); rStr.trim();
          if (rStr != "null") newR = rStr.toFloat();
        }
      }
      // Cari E
      int pE = resp.indexOf("\"E\":", pLimits);
      if (pE >= 0) {
        int s = pE + 4;
        int e = resp.indexOf(',', s);
        if (e < 0) e = resp.indexOf('}', s);
        if (e > s) {
          String eStr = resp.substring(s, e); eStr.trim();
          if (eStr != "null") newE = eStr.toFloat();
        }
      }
      // Cari O
      int pO = resp.indexOf("\"O\":", pLimits);
      if (pO >= 0) {
        int s = pO + 4;
        int e = resp.indexOf(',', s);
        if (e < 0) e = resp.indexOf('}', s);
        if (e > s) {
          String oStr = resp.substring(s, e); oStr.trim();
          if (oStr != "null") newO = oStr.toFloat();
        }
      }

      bool changed = false;
      if (!isnan(newR) && fabs(newR - setR) > 0.001) { setR = newR; changed = true; }
      if (!isnan(newE) && fabs(newE - setE) > 0.001) { setE = newE; changed = true; }
      if (!isnan(newO) && fabs(newO - setO) > 0.001) { setO = newO; changed = true; }
      if (changed) {
        saveSetpoints();
        Serial.printf("Setpoints updated from backend: R=%.2f E=%.2f O=%.2f\n", setR, setE, setO);
      }
    }
  }
  http.end();
}

// Auto OFF GPIO2 (pulse only). IMPORTANT: do NOT end defrost here.
void updateGpio2AutoOffAndResetServer(){
  if(!defrozActive) return;

  // Matikan GPIO2 setelah 2 detik (pulse), tapi jangan ubah currentState/defrozActive.
  if (millis() - defrozStart >= 2000 && digitalRead(GPIO2_PIN) == LOW) {
    digitalWrite(GPIO2_PIN, HIGH);
    Serial.println("GPIO2 HIGH auto reset (pulse selesai)");

    // Kirim balik ke server agar server state kembali 0
    if (WiFi.status() == WL_CONNECTED) {
      HTTPClient http;
      http.begin(String(SERVER_URL) + "state.php");
      http.addHeader("Content-Type", "application/x-www-form-urlencoded");
      http.POST("state=0");
      http.end();
    }
  }

  // NOTE: defrozActive dan currentState dibiarkan sampai handleSystemState() menyelesaikannya.
}

// ------------------ Serial Commands ------------------
void handleSerialCommands(){
  static String serialLine = "";
  while (Serial.available() > 0) {
    char c = (char)Serial.read();
    if (c == '\r' || c == '\n') {
      if (serialLine.length() > 0) {
        String cmd = serialLine; serialLine = "";
        cmd.trim();
        if (cmd.equalsIgnoreCase("DEFROST")) {
          if (!defrozActive && currentState == RUNNING) {
            currentState = DEFROST;
            noInterrupts(); jam=0; menit=0; detik=0; interrupts();
            Serial.println("DEFROST manual dari Serial");

            // reset logic & mulai defrost
            resetRelayLogic(true);
            digitalWrite(Rl2, HIGH); // heater ON
            
            defrozStart = millis();
            defrozActive = true;
          } else {
            Serial.println("DEFROST diabaikan: sudah aktif atau bukan RUNNING");
          }
        }
      }
    } else {
      serialLine += c;
      if (serialLine.length() > 128) {
        serialLine.remove(0, serialLine.length() - 128);
      }
    }
  }
}
